# from http://rstudio-pubs-static.s3.amazonaws.com/3336_03636030d93d47de9131e625b72f58c6.html
library(lme4)
library(ggplot2)
library(reshape2)
theme_set(theme_bw())
set.seed(102)
n_indiv <- 200
n_per_indiv <- 4
n_tot <- n_indiv * n_per_indiv
id <- gl(n_indiv, n_per_indiv)  
av_wealth <- rlnorm(n_indiv, 0, 1)  ## avg wealth of individuals
ac_wealth <- av_wealth[id] + rlnorm(n_tot, 0, 1)  ## wealth in each year:
av_ratio <- rbeta(n_indiv, 10, 10)  ## 50/50, close to Gaussian
## car/holiday ratio by year (larger shape parameter/less variability)
ac_ratio <- rbeta(n_tot, 2 * av_ratio[id], 2 * (1 - av_ratio[id]))
y.car <- (ac_wealth * ac_ratio)^0.25
y.hol <- (ac_wealth * (1 - ac_ratio))^0.25
Spending <- data.frame(y.hol, y.car, id)
qplot(y.hol, y.car, data = Spending) + geom_smooth(method = "lm")


# naive regression
m0 <- lm(y.car ~ y.hol, data = Spending)
m0

# mcmcglmm
m1_id <- MCMCglmm(y.car ~ y.hol, random = ~id, data = Spending, verbose = FALSE)
m1_idyr <- MCMCglmm(cbind(y.hol, y.car) ~ trait - 1, 
                    random = ~us(trait):id, 
                    rcov = ~us(trait):units, data = Spending, 
                    family = c("gaussian", "gaussian"), 
                    verbose = FALSE)
summary(m1_id)
summary(m1_idyr)


# Compute regression coefficients for car as a function of holiday (ratios of car-holiday covariance to holiday variance at each level):
  
rd <- as.data.frame(m1_idyr$VCV)
id.regression <- with(rd, `y.car:y.hol.id`/`y.hol:y.hol.id`)
units.regression <- with(rd, `y.car:y.hol.units`/`y.hol:y.hol.units`)
res_m1_id <- setNames(summary(m1_id)$solutions[2, 1:3], c("est", "lwr", "upr"))
res_m1_idyr_id <- c(est = mean(id.regression), 
                    setNames(quantile(id.regression, 
                    c(0.025, 0.975)), c("lwr", "upr")))
res_m1_idyr_units <- c(est = mean(units.regression), 
                       setNames(quantile(units.regression, 
                      c(0.025, 0.975)), c("lwr", "upr")))

ff <- function(x) factor(x, levels = unique(x))
combdat <- data.frame(param = ff(c("naive", "m_L1", "m_L2.units", "m_L2.id")), 
              type = ff(c("lm", rep("MCMCglmm", 3))), 
                      rbind(res_reg0, 
                            res_m1_id, 
                            res_m1_idyr_units, 
                            res_m1_idyr_id))
# Plot results so far:
(g1 <- ggplot(combdat, aes(param, est, ymin = lwr, ymax = upr, colour = type)) + 
   geom_pointrange() + labs(x = "", y = "estimate") + 
   geom_hline(yintercept = 0, colour = "black", lwd = 2, alpha = 0.2) + 
   coord_flip() + 
   scale_colour_brewer(palette = "Dark2"))




######## lme4

# Now try to do this with lme4 – partly for the challenge, partly for 
# the Bayes-phobic, and possibly for other advantages (e.g. speed, 
# for very large data sets?)

# The first thing we need to do is melt the data: this is done implicitly 
# by MCMCglmm's multi-trait framework, but isn't too hard. We add a variable 
# to index the observations in the original data set (i.e. person-years) 
# and give the resulting derived variable the name trait for consistency 
# with MCMCglmm …

mSpending <- melt(data.frame(Spending, obs = seq(nrow(Spending))), 
                  id.var = c("obs", "id"), variable.name = "trait")
# The single-level model is easy:
_id <- lmer(y.car ~ y.hol + (1 | id), data = Spending)