library(plyr)
library(mvtnorm)
library(lme4)
library(MCMCglmm)
# what does rmvnorm() do?
corvarm = matrix(c(1,.3,.3,1), ncol=2)
plot(rmvnorm(2000, sigma=corvarm))



## ------------------------ start example -----------------------------

# based on Dave Kleinschmidt's data generator
# extended to bivariate normal data
make.data.generator <- function(true.effects=matrix(rep(0,n.pred*ncol(dep.var)), ncol=n.pred),
                                resid.var=1,
                                dep.var=diag(rep(resid.var, n.depvar)),
                                ranef.var=diag(rep(1, ncol(dep.var) * ncol(true.effects))),
                                n.subj=24,
                                n.obs=24,
                                n.pred=2,
                                n.depvar=2
)
{
  # create design matrix for our made up experiment
  # this part would need to be generalized to allow more than two predictors
  # (more than intercept plus Pred1)
  data.str <- data.frame(Pred1=factor(c(rep('high', n.obs/2), rep('low', n.obs/2))))
  contrasts(data.str$Pred1) <- contr.sum(2)
  model.mat <- model.matrix(~ 1 + Pred1, data.str)
  # below this point it should be work even for more than 2 predictors
  
  generate.data <- function() {
    # sample data set under mixed effects model with random slope/intercepts
    simulated.data <- rdply(n.subj, {
      beta <- matrix(t(rmvnorm(n=1, sigma=ranef.var)), ncol=ncol(true.effects)) + true.effects
      expected.outcome <- model.mat %*% beta
      epsilon <- rmvnorm(n=nrow(expected.outcome), mean=rep(0, ncol(dep.var)), sigma=sqrt(dep.var))
      data.frame(cbind(data.str,
                 data.frame(outcome=expected.outcome + epsilon)
      ))
    })
    names(simulated.data)[1] <- 'Subject'
    simulated.data
  }
}

gen.dat <- make.data.generator()
vglm(cbind(outcome.1,outcome.2) ~ Pred1, family = gaussianff, data = gen.dat())

fit.models = function(data) {
  # fit model and extract coefs
  vglm.coefs <- coefficients(summary(
    vglm(cbind(outcome.1,outcome.2) ~ Pred1, family = gaussianff, data = data)
    ))[, 1:3]
  
  mcmcglmm.coefs <- summary(
    MCMCglmm(cbind(outcome.1,outcome.2) ~ trait*Pred1 - 1 - Pred1, 
          random = ~us(trait):Subject, 
          rcov = ~us(trait):units, data = data, 
          family = c("gaussian", "gaussian"), 
          verbose = FALSE)
  )$solutions[,1:3]

  mcmcglmm.coefs[,2] = 
    ((mcmcglmm.coefs[,1] - mcmcglmm.coefs[,2]) +
    (mcmcglmm.coefs[,3] - mcmcglmm.coefs[,1])) / (2 * 1.96)
  mcmcglmm.coefs[,3] = mcmcglmm.coefs[,1] / mcmcglmm.coefs[,2]
  colnames(mcmcglmm.coefs) = colnames(vglm.coefs)
  rownames(mcmcglmm.coefs) = rownames(vglm.coefs)
  
  # format output all pretty
  rbind(
    data.frame(model='vglm', predictor=rownames(vglm.coefs), vglm.coefs),
    data.frame(model='mcmcglmm', predictor=rownames(vglm.coefs), mcmcglmm.coefs)
  )
}

simulations = rdply(.n=100,
                  fit.models(gen.dat()),
                  .progress='text')

# evaluation
daply(simulations, .(model, predictor), function(df) type1err=mean(abs(df$z.value)>1.96))
ggplot(simulations, aes(x=z.value, color=model)) +
  geom_vline(xintercept=c(-1.96, 1.96), color='#888888', linetype=3) +
  scale_x_continuous('z value') +
  geom_density() +
  facet_grid(predictor~.)


# with effect
gen.dat <- make.data.generator(true.effects = matrix(c(0,2,10,4), ncol=2))
simulations.wEffect = rdply(.n=100,
      fit.models(gen.dat()),
      .progress='text')
daply(simulations.wEffect, .(model, predictor), function(df) 
  type1err=mean(abs(df$z.value)>1.96))
ggplot(simulations.wEffect, aes(x=Estimate, color=model)) +
  geom_vline(xintercept=c(-1.96, 1.96), color='#888888', linetype=3) +
  scale_x_continuous('Estimate') +
  geom_density() +
  facet_grid(predictor~.)

# with effect and covariance of dependent variable
gen.dat <- make.data.generator(
  true.effects = matrix(c(0,2,10,4), ncol=2),
  dep.var = matrix(c(1,.3,.3,1), ncol=2)
)
simulations.wEffectCov = rdply(.n=100,
      fit.models(gen.dat()),
      .progress='text')
daply(simulations.wEffectCov, .(model, predictor), 
      function(df) type1err=mean(abs(df$z.value)>1.96))
ggplot(simulations.wEffectCov, aes(x=z.value, color=model)) +
  geom_vline(xintercept=c(-1.96, 1.96), color='#888888', linetype=3) +
  scale_x_continuous('z value') +
  geom_density() +
  facet_grid(predictor~.)
